' Parameterized Fractal Tree
' Rev 1.0.0 William M Leue, 3-May-2024

option default integer
option base 1
option angle degrees

' Constants
const MIN_NBRANCHES = 2
const MAX_NBRANCHES = 10
const RATIO         = 0.62
const STEM_LEN      = mm.vres\3
const STEM_THK      = 3
const MIN_ORDER     = 1
const MAX_ORDER     = 10
const NGRADIENTS    = 6

' Globals
dim float sl = STEM_LEN
dim float nbranches = 0
dim corder   = 0
dim sx       = mm.hres\2
dim sy       = mm.vres-1
dim float hue = 0.0
dim float sat = 1.0
dim float brt = 1.0
dim float start_hue = 0.0
dim float hue_inc = 0.0
dim float ubrangle = 0.0
dim float utangle = 0.0
dim float ustangle = 0.0
dim gradient = 1
dim float grad_pars(NGRADIENTS, 2)
dim float brtable(MAX_NBRANCHES)

' Main Program
'open "debug.txt" for output as #1
ReadParams
do
  do
    ok = 1
    cls
    print "Enter the Number of Branches per Node in the Tree (2-10) [0 to quit]: ";
    input "", a$
    nbranches = val(a$)
    if nbranches = 0 then end
    if (nbranches < MIN_NBRANCHES) or (nbranches > MAX_NBRANCHES) then
      ok = 0
      print "** value out of range -- please try again **"
      pause 1000
    end if
  loop until ok
  do
    ok = 1
    print "Enter the order of the Tree (1-10): ";
    input "", a$
    corder = val(a$)
    if (corder < MIN_ORDER) or (corder > MAX_ORDER) then
      ok = 0
      print "*** value out of range -- please try again **"
      pause 1000
    end if
  loop until ok
  do
    ok = 1
    print "Enter the color gradient type:"
    print "  1. Red -> Yellow"
    print "  2. Green -> Blue"
    print "  3. Blue -> Violet"
    print "  4. Yellow -> Red"
    print "  5. Blue -> Green"
    print "  6. Red -> Green"
    print "Your choice (1-6): ";
    input "", a$
    gradient = val(a$)
    if (gradient < 1) or (gradient > 6) then
      ok = 0
      print "*** value out of range -- please try again **"
      pause 1000
    end if
  loop until ok
  cls
  start_hue = grad_pars(gradient, 1)
  hue_inc = grad_pars(gradient, 2)/MAX_ORDER
  ubrangle = brtable(nbranches)
  utangle = (nbranches-1)*ubrangle
  ustangle = 0.5*utangle
  HSV2RGB start_hue, sat, brt, r, g, b
  c = rgb(r, g, b)
  DrawStem sx, sy, sl, c
  DrawBranches sx, sy-sl, sl, 90.0, corder
  m$ = "Order " + str$(corder) + " Tree with " + str$(nbranches) + " Branches"
  text mm.hres\2, 0, m$, "CT", 4,, rgb(green)
  text mm.hres\2, mm.vres-1, "Press Any Key to Continue", "CB"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
loop
end

' Read the tables of branch angles and color gradients
sub ReadParams
  local i. j
  for i = 1 to MAX_NBRANCHES
    read brtable(i)
  next i
  for i = 1 to NGRADIENTS
    for j = 1 to 2
      read grad_pars(i, j)
    next j
  next i
end sub

' Draw the start stem
sub DrawStem sx, sy, sl, c
  line sx-1, sy, sx-1, sy-sl, STEM_THK, c
end sub

' Draw the branches (recursive!)
sub DrawBranches sx as float, sy as float, sl as float, angle as float, corder
  local c, i, r, g, b
  local float brl, bangle, x, y
  hue = start_hue + (MAX_ORDER-corder+1)*hue_inc
  HSV2RGB hue, sat, brt, r, g, b
  c = rgb(r, g, b)
  brl = sl*RATIO
  bangle = angle + ustangle
  for i = 1 to nbranches
    x = sx + brl*cos(bangle)
    y = sy - brl*sin(bangle)
    line sx, sy, x, y,, c
    if corder > 1 then
      DrawBranches x, y, brl, bangle, corder-1
    end if
    inc bangle, -ubrangle
  next i
end sub

' Convert an HSV value to its RGB equivalent
' The S and V values must be in range 0..1; the H value must
`' be in range 0..360. The RGB values will be in range 0..255.
sub HSV2RGB h as float, s as float, v as float, r, g, b
  local float i, hh, f, p, q, t, x, c, rp, gp, bp
  c = v*s
  hh = h/60.0
  i = int(hh)
  f = hh - i
  p = v*(1-s)
  q = v*(1-s*f)
  t = v*(1-s*(1-f))
  x = c*(1.0 - hh MOD 2 - 1)
  select case i
    case 0
      rp = v : gp = t : bp = p
    case 1
      rp = q : gp = v : bp = p
    case 2
      rp = p : gp = v : bp = t
    case 3
      rp = p : gp = q : bp = v
    case 4
      rp = t : gp = p : bp = v
    case 5
      rp = v : gp = p : bp = q
  end select
  r = rp*255.0 : g = gp*255.0 : b = bp*255.0
end sub

' Branch Angles per number of branches
data 0.0, 90.0, 60.0, 45.0, 22.5, 30.0, 25.0, 22.5, 20.0, 18.0

' Color Gradient Type Data
' Start Hue, Signed Hue Increment
data 0.0,    60.0
data 120.0,  120.0
data 240.0,  60.0
data 60.0,  -60.0
data 240.0, -120.0
data 0.0,    120.0
